using System;
using System.Xml;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class holds user name information returned by remote VistALink server 
	/// (M listener, .NET listener) in response to VistALink get user demographics request. 
	/// </summary>
	public class RemoteUserNameInfo
	{
		private readonly string	_namePrefix, _firstName, _middleName, _lastName, _nameSuffix, 
								_degree, _standardConcatenatedName, _newPerson01Name;

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_NAME_INFO_NODE_NAME = "NameInfo";
		private const string XMLCONSTS_NAME_PREFIX_NI_ATTRIBUTE_NAME = "prefix";
		private const string XMLCONSTS_NAME_GIVEN_FIRST_NI_ATTRIBUTE_NAME = "givenFirst";
		private const string XMLCONSTS_NAME_MIDDLE_NI_ATTRIBUTE_NAME = "middle";
		private const string XMLCONSTS_NAME_FAMILY_LAST_NI_ATTRIBUTE_NAME = "familyLast";
		private const string XMLCONSTS_NAME_SUFFIX_NI_ATTRIBUTE_NAME = "suffix";
		private const string XMLCONSTS_NAME_NEWPERSON01NAME_NI_ATTRIBUTE_NAME = "newPerson01Name";
		private const string XMLCONSTS_NAME_DEGREE_NI_ATTRIBUTE_NAME = "degree";
		private const string XMLCONSTS_NAME_STANDARDCONCATENATED_NI_ATTRIBUTE_NAME = "standardConcatenated";

		/// <summary>
		/// This constructor specifies remote user name information for VistA systems.
		/// All parameters are required.
		/// </summary>
		/// <param name="namePrefix">Name prefix (Mr., Mrs., etc.).</param>
		/// <param name="firstName">First (given) name.</param>
		/// <param name="middleName">Middle name.</param>
		/// <param name="lastName">Last (family) name.</param>
		/// <param name="nameSuffix">Name suffix.</param>
		/// <param name="newPerson01Name">Name of person stored in the .01 field of the NEW PERSON file (#200) in VistA.</param>
		/// <param name="degree">User scientific degree.</param>
		/// <param name="standardConcatenatedName">
		/// Standard concatenated name. VistALink protocol 
		/// defines this parameter but does not describe it.
		/// </param>
		public RemoteUserNameInfo( 
			string namePrefix, 
			string firstName, 
			string middleName, 
			string lastName, 
			string nameSuffix, 
			string newPerson01Name, 
			string degree, 
			string standardConcatenatedName )
		{
			if( namePrefix == null )
				throw( new ArgumentNullException( "namePrefix" ) );

			if( firstName == null )
				throw( new ArgumentNullException( "firstName" ) );

			if( middleName == null )
				throw( new ArgumentNullException( "middleName" ) );

			if( lastName == null )
				throw( new ArgumentNullException( "lastName" ) );

			if( nameSuffix == null )
				throw( new ArgumentNullException( "nameSuffix" ) );

			if( newPerson01Name == null )
				throw( new ArgumentNullException( "newPerson01Name" ) );

			if( degree == null )
				throw( new ArgumentNullException( "degree" ) );

			if( standardConcatenatedName == null )
				throw( new ArgumentNullException( "standardConcatenatedName" ) );

			_namePrefix = namePrefix;
			_firstName = firstName;
			_middleName = middleName;
			_lastName = lastName;
			_nameSuffix = nameSuffix;
			_newPerson01Name = newPerson01Name;
			_degree = degree;
			_standardConcatenatedName = standardConcatenatedName;
		}

		// TODO: Add unit tests. 
		/// <summary>
		/// XML deserialization factory method retrieving a remote user name information XML element 
		/// from a list of descendants of a given node by its unique tag name, parsing it
		/// and returning remote user name info object created from it.
		/// </summary>
		/// <param name="rootElement">XML node containing remote user name info element among its children.</param>
		/// <returns>Remote user info object deserialized from XML.</returns>
		public static RemoteUserNameInfo ParseFindChildNodeByUniqueTagName( XmlElement rootElement )
		{
			if( rootElement == null )
				throw( new ArgumentNullException( "rootElement" ) );

			return Parse( XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_NAME_INFO_NODE_NAME ) );
		}

		/// <summary>
		/// XML deserialization factory method parsing supplied remote user name information 
		/// XML element and returning remote user name info object created from it.
		/// </summary>
		/// <param name="nameInfoNode">XML node containing remote user name info.</param>
		/// <returns>Remote user name info object deserialized from XML.</returns>
		public static RemoteUserNameInfo Parse( XmlNode nameInfoNode )
		{
			XmlElement _nameInfoElement = XmlUtility.ParseValidateConvertNodeToElement( nameInfoNode, XMLCONSTS_NAME_INFO_NODE_NAME );
			
			return new RemoteUserNameInfo( 
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_PREFIX_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_GIVEN_FIRST_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_MIDDLE_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_FAMILY_LAST_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_SUFFIX_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_NEWPERSON01NAME_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_DEGREE_NI_ATTRIBUTE_NAME ),
				XmlUtility.ParseGetRequiredAttributeValue( _nameInfoElement, XMLCONSTS_NAME_STANDARDCONCATENATED_NI_ATTRIBUTE_NAME ) );
		}

		/// <summary>
		/// XML serialization method writing out remote user 
		/// name info to suppied XML writer.
		/// </summary>
		/// <param name="writer">XmlWriter to write to.</param>
		public void WriteNameInfoToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );
		
			writer.WriteStartElement( XMLCONSTS_NAME_INFO_NODE_NAME );
			
				writer.WriteAttributeString( XMLCONSTS_NAME_PREFIX_NI_ATTRIBUTE_NAME, _namePrefix );
				writer.WriteAttributeString( XMLCONSTS_NAME_GIVEN_FIRST_NI_ATTRIBUTE_NAME, _firstName );
				writer.WriteAttributeString( XMLCONSTS_NAME_MIDDLE_NI_ATTRIBUTE_NAME, _middleName );
				writer.WriteAttributeString( XMLCONSTS_NAME_FAMILY_LAST_NI_ATTRIBUTE_NAME, _lastName );
				writer.WriteAttributeString( XMLCONSTS_NAME_SUFFIX_NI_ATTRIBUTE_NAME, _nameSuffix );
				writer.WriteAttributeString( XMLCONSTS_NAME_NEWPERSON01NAME_NI_ATTRIBUTE_NAME, _newPerson01Name );
				writer.WriteAttributeString( XMLCONSTS_NAME_DEGREE_NI_ATTRIBUTE_NAME, _degree );
				writer.WriteAttributeString( XMLCONSTS_NAME_STANDARDCONCATENATED_NI_ATTRIBUTE_NAME, _standardConcatenatedName );

			writer.WriteEndElement();
		}

		/// <summary>
		/// Name prefix (Mr., Mrs., etc.).
		/// </summary>
		public string NamePrefix
		{
			get
			{
				return _namePrefix;
			}
		}

		/// <summary>
		/// First/given Name
		/// </summary>
		public string FirstName
		{
			get
			{
				return _firstName;
			}
		}

		/// <summary>
		/// Middle name.
		/// </summary>
		public string MiddleName
		{
			get
			{
				return _middleName;
			}
		}

		/// <summary>
		/// Last/family Name
		/// </summary>
		public string LastName
		{
			get
			{
				return _lastName;
			}
		}

		/// <summary>
		/// Name suffix (Jr, Sr, etc.)
		/// </summary>
		public string NameSuffix
		{
			get
			{
				return _nameSuffix;
			}
		}

		/// <summary>
		/// Name of person stored in the .01 field of the NEW PERSON file (#200) in VistA.
		/// </summary>
		public string NewPerson01Name
		{
			get
			{
				return _newPerson01Name;
			}
		}

		/// <summary>
		/// User degree.
		/// </summary>
		public string Degree
		{
			get
			{
				return _degree;
			}
		}

		/// <summary>
		/// Standard concatenated name. VistALink protocol defines this 
		/// parameter but does not describe it.
		/// </summary>
		public string StandardConcatenatedName
		{
			get
			{
				return _standardConcatenatedName;
			}
		}
	}
}
